/** @file   settings.cpp
 * @brief   Implementation of Settings - class.
 * @version $Revision: 1.11 $
 * @author  Tomi Lamminsaari
 */

#include "settings.h"
#include "weapon.h"
#include "filehasher.h"
#include "www_assert.h"
#include "bonusobject.h"
using std::string;
using std::endl;


namespace WeWantWar {



//********************************************************************
//                                                                   *
//      Static members and constants                                 *
//                                                                   *
//********************************************************************

const string Settings::SAVE_SETTINGS_TO_FILE = "data/gamesettings.cfg";
const int Settings::CONTROLMODE1;
const int Settings::CONTROLMODE2;

int Settings::controlMode = Settings::CONTROLMODE1;
int Settings::key_up = KEY_W;
int Settings::key_down = KEY_S;
int Settings::key_left = KEY_A;
int Settings::key_right = KEY_D;
int Settings::key_grenade = KEY_F;
int Settings::key_detonate = KEY_SPACE;

int Settings::firstaid_points = 50;

bool Settings::wallShadows = true;
bool Settings::lightSources = true;
bool Settings::shootingLights = true;
bool Settings::explosionLights = true;

// Audio settings
bool Settings::titleMusicOn = true;
bool Settings::musicOn = true;
bool Settings::soundsOn = true;

int Settings::startFromLevel = 0;
eng2d::FileList Settings::levelList;

const int Settings::KEasy;
const int Settings::KNormal;
const int Settings::KHard;

int Settings::difficultyLevel = 1;
int Settings::lightbeamMatrixCellSize = 8;

std::vector< Utils::GObjectConfig > Settings::gameObjectSettings;
std::vector< Utils::WeaponSpecsConfig > Settings::weaponSpecs;

std::vector< std::string > Settings::bonusObjectNames;


///
/// Scores
/// ======

int Settings::SCORE_BULLETHIT = 2;
int Settings::SCORE_SHOOT = -1;
int Settings::SCORE_CARNIVOREALIEN = 200;
int Settings::SCORE_SMALLWORMALIEN = 100;
int Settings::SCORE_WINGEDALIEN = 150;
int Settings::SCORE_PROCTORALIEN = 500;
int Settings::SCORE_MINIGUNALIEN = 600;
int Settings::SCORE_PREDATORALIEN = 400;


///
/// Cheats
/// ======

bool Settings::cheatLifes = false;
bool Settings::cheatAmmo = false;
bool Settings::cheatYouko = false;

bool Settings::detonateDown = false;

/** Loads the settings
 */
int Settings::init()
{
  // Load the gameobject and Weapon settings.
  if ( loadSettings() != 0 ) {
    return -1;
  }
  
  initBonusObjects();
  
  // Load the settings of the Settings-menu
  std::ifstream fin( SAVE_SETTINGS_TO_FILE.c_str() );
  if ( (!fin) == false ) {
    // Read the settings
    string tmp;
    fin >> tmp;
    if ( tmp == "<wewantwar_general_settings>" ) {
      while ( true ) {
        if ( fin.eof() == true ) {
          break;
        }
        fin >> tmp;
        if ( tmp == "#" ) {
          fin.ignore(4096, '\n');
        } else if ( tmp == "controlmode:" ) {
          fin >> controlMode;
        } else if ( tmp == "musics:" ) {
          fin >> musicOn;
        } else if ( tmp == "titlemusic:" ) {
          fin >> titleMusicOn;
        } else if ( tmp == "sounds:" ) {
          fin >> soundsOn;
        } else if ( tmp == "key_up:" ) {
          fin >> key_up;
        } else if ( tmp == "key_down:" ) {
          fin >> key_down;
        } else if ( tmp == "key_left:" ) {
          fin >> key_left;
        } else if ( tmp == "key_right:" ) {
          fin >> key_right;
        } else if ( tmp == "key_grenade:" ) {
          fin >> key_grenade;
        } else if ( tmp == "start_from:" ) {
          fin >> startFromLevel;
        } else if ( tmp == "difficulty:" ) {
          fin >> difficultyLevel;
        } else if ( tmp == "lights_staticlights:" ) {
          fin >> lightSources;
        } else if ( tmp == "lights_wallshadows:" ) {
          fin >> wallShadows;
        } else if ( tmp == "lights_gunshot:" ) {
          fin >> shootingLights;
        } else if ( tmp == "lights_explosions:" ) {
          fin >> explosionLights;
        } else if ( tmp == "cheat_lifes:" ) {
          fin >> cheatLifes;
        } else if ( tmp == "cheat_ammo:" ) {
          fin >> cheatAmmo;
        } else if ( tmp == "cheat_immortal_Youko:" ) {
          fin >> cheatYouko;
        } else if ( tmp == "lightbeam_matrix:" ) {
          fin >> lightbeamMatrixCellSize;
        } else if ( tmp == "</wewantwar_general_settings>" ) {
          break;
        }
      }
    }
  }
  fin.close();
  
  if ( levelList.load( "maps/level_files.txt" ) != 0 ) {
    return -1;
  }
  return 0;
}



/** Saves the general settings to configuration file.
 */
void Settings::saveGeneralSettings()
{
  std::ofstream fout( SAVE_SETTINGS_TO_FILE.c_str() );
  if ( !fout ) {
    return;
  }
  string indent = "    ";
  fout << "<wewantwar_general_settings>" << endl;
  fout << indent << "# " << endl;
  fout << indent << "# Automatically generated by the game." << endl;
  fout << indent << "# DO NOT MODIFY !" << endl;
  fout << indent << "# " << endl;
  fout << indent << "controlmode: " << controlMode << endl;
  fout << indent << "musics: " << musicOn << endl;
  fout << indent << "titlemusic: " << titleMusicOn << endl;
  fout << indent << "sounds: " << soundsOn << endl;
  fout << indent << "key_up: " << key_up << endl;
  fout << indent << "key_down: " << key_down << endl;
  fout << indent << "key_left: " << key_left << endl;
  fout << indent << "key_right: " << key_right << endl;
  fout << indent << "key_grenade: " << key_grenade << endl;
  fout << indent << "start_from: " << startFromLevel << endl;
  fout << indent << "difficulty: " << difficultyLevel << endl;
  fout << indent << "lightbeam_matrix: " << lightbeamMatrixCellSize << endl;
  fout << endl;
  fout << indent << "lights_staticlights: " << lightSources << endl;
  fout << indent << "lights_wallshadows: " << wallShadows << endl;
  fout << indent << "lights_gunshot: " << shootingLights << endl;
  fout << indent << "lights_explosions: " << explosionLights << endl;
  fout << endl;
  fout << indent << "cheat_lifes: " << cheatLifes << endl;
  fout << indent << "cheat_ammo: " << cheatAmmo << endl;
  fout << indent << "cheat_immortal_Youko: " << cheatYouko << endl;
  fout << "</wewantwar_general_settings>" << endl;
  fout.close();
}



/** Returns the properties for the GameObjects
 */
std::string Settings::getObjectProperty( ObjectID::Type objectID,
                                  const string& propName )
                                  throw ( Exception )
{
  for ( int i=0; i < gameObjectSettings.size(); i++ ) {
    if ( gameObjectSettings.at(i).objectID == objectID ) {
      std::map< std::string, std::string >& rP = gameObjectSettings.at(i).properties;
      std::map< std::string, std::string >::iterator it = rP.find( propName );
      if ( it == rP.end() ) {
        throw Exception( string("invalid object property") + propName );
      }
      return it->second;
    }
  }
}


/** Returns the requested property as integer.
 */
int Settings::intObjProp( ObjectID::Type oid, const string& propName )
                        throw ( Exception )
{
  return atoi( getObjectProperty( oid, propName ).c_str() );
}



/** Returns the requested property as float
 */
float Settings::floatObjProp( ObjectID::Type oid, const string& propName )
                        throw ( Exception )
{
  return atof( getObjectProperty( oid, propName ).c_str() );
}



/** Returns the name of the bonusobject.
 */
std::string Settings::getBonusObjectName( int bonusID )
{
  std::string bonusname = "";
  switch ( bonusID ) {
    case ( BonusObject::RIFLE ): {
      bonusname = "RIFLE";
      break;
    }
    case ( BonusObject::SHOTGUN ): {
      bonusname = "SHOTGUN";
      break;
    }
    case ( BonusObject::GRENADE ): {
      bonusname = "GRENADES";
      break;
    }
    case ( BonusObject::FIRSTAID ): {
      bonusname = "FIRST AID";
      break;
    }
    case ( BonusObject::FLAMETHROWER ): {
      bonusname = "FLAMETHROWER";
      break;
    }
    case ( BonusObject::MINIGUN ): {
      bonusname = "MINIGUN";
      break;
    }
    case ( BonusObject::UZI ): {
      bonusname = "DUAL UZIS";
      break;
    }
    case ( BonusObject::EXTRALIFE ): {
      bonusname = "EXTRA LIFE";
      break;
    }
    case ( BonusObject::SCORE_100PTS ): {
      bonusname = "100 POINTS";
      break;
    }
    case ( BonusObject::SCORE_200PTS ): {
      bonusname = "200 POINTS";
      break;
    }
    case ( BonusObject::SNIPERRIFLE ): {
      bonusname = "SNIPER RIFLE";
      break;
    }
    case ( BonusObject::ROCKETLAUNCHER ): {
      bonusname = "ROCKET LAUNCHER";
      break;
    }
    case ( BonusObject::BODYARMOR ): {
      bonusname = "BODY ARMOR";
      break;
    }
    default: {
      bonusname = "UNNAMED";
      break;
    }
  }
  return bonusname;
}



/** Returns the Weapon specs
 */
std::string Settings::getWProp( int wid, const string& propName )
                        throw ( Exception )
{
  for ( int i=0; i < weaponSpecs.size(); i++ ) {
    if ( weaponSpecs.at(i).weaponTypeID == wid ) {
      std::map< std::string, std::string >& rP = weaponSpecs.at(i).properties;
      std::map< std::string, std::string >::iterator it = rP.find( propName );
      if ( it == rP.end() ) {
        throw Exception( "Invalid weapon property" );
      }
      return it->second;
    }
  }
}



/** Returns the Weapon specs as an integer
 */
int Settings::intWProp( int wid, const string& propName )
                        throw ( Exception )
{
  for ( int i=0; i < weaponSpecs.size(); i++ ) {
    if ( weaponSpecs.at(i).weaponTypeID == wid ) {
      std::map< std::string, std::string >& rP = weaponSpecs.at(i).properties;
      std::map< std::string, std::string >::iterator it = rP.find( propName );
      if ( it == rP.end() ) {
        throw Exception( "Invalid weapon property" );
      }
      return atoi( (it->second).c_str() );
    }
  }
}



/** Returns the Weapon specs as float
 */
float Settings::floatWProp( int wid, const string& propName )
                        throw ( Exception )
{
  for ( int i=0; i < weaponSpecs.size(); i++ ) {
    if ( weaponSpecs.at(i).weaponTypeID == wid ) {
      std::map< std::string, std::string >& rP = weaponSpecs.at(i).properties;
      std::map< std::string, std::string >::iterator it = rP.find( propName );
      if ( it == rP.end() ) {
        throw Exception( "Invalid weapon property" );
      }
      return atof( (it->second).c_str() );
    }
  }
}

/** Loads the settings from configuration file
 */
int Settings::loadSettings()
{
  eng2d::FileHasher hf( "data/config.txt" );
  bool hashFailure = hf.check();

  std::ifstream fin( "data/config.txt" );
  if ( !fin ) {
    return -1;
  }

  std::string tmp;
  fin >> tmp;
  if ( tmp != "<wewantwar_settings>" ) {
    // Invalid settings file.
    alert( "Invalid configuration file", 0,0, "ok",0, 0,0 );
    return -1;
  }

  // Check the Hashcode.

  if ( hashFailure == true ) {
    alert( "Invalid configuration file",
           "Hashcode is invalid",
           "You've been hacking, aren't you? :)", "ok",0, 0,0);
    return -1;
  }

  
  int ret = readElement_Settings( fin );
  if ( ret != 0 ) {
    fin.close();
    return -1;
  }
  fin.close();
  return 0;
}



/** Reads the contents of the Settings-element.
 */
int Settings::readElement_Settings( std::ifstream& rFin )
{
  LOG_MESSAGE( "        Reading SETTINGS-element..." );
  std::string tmp;
  
  while ( true ) {
    if ( rFin.eof() == true ) {
      alert( "EOF encountered", "data/config.txt", 0, "ok",0, 0,0 );
      return -1;
    }
    
    rFin >> tmp;
    if ( tmp == "</wewantwar_settings>" ) {
      break;
      
    } else if ( tmp == "#" ) {
      rFin.ignore( 4096, '\n' );
      
    } else if ( tmp == "<score>" ) {
      int ret = readElement_Score( rFin );
      if ( ret != 0 ) {
        return -1;
      }
      
    } else if ( tmp == "<weapons>" ) {
      int ret = readElement_Weapons( rFin );
      if ( ret != 0 ) {
        return -1;
      }
      
    } else if ( tmp == "<gameobjects>" ) {
      int ret = readElement_GameObjects( rFin );
      if ( ret != 0 ) {
        return -1;
      }
      
    }
      
  }
  LOG_MESSAGE( "        done" );
  return 0;
}



/** Reads the SCORE - element
 */
int Settings::readElement_Score( std::istream& rIn )
{
  LOG_MESSAGE("            Reading SCORE - element..." );
  LOG_MESSAGE("            done" );
  return 0;
}



/** Reads the GAMEOBJECTS - element
 */
int Settings::readElement_GameObjects( std::istream& rIn )
{
  LOG_MESSAGE("            Reading GAMEOBJECTS-element..." );
  // Create the lookup table for converting the tag-names to ObjectID::Type -
  // values.
  std::map< std::string, ObjectID::Type > goMapping;
  goMapping["<player>"] = ObjectID::TYPE_PLAYER;
  goMapping["<carnivorealien>"] = ObjectID::TYPE_CARNIVOREALIEN;
  goMapping["<smallwormalien>"] = ObjectID::TYPE_SMALLWORMALIEN;
  goMapping["<wingedalien>"] = ObjectID::TYPE_WINGEDALIEN;
  goMapping["<proctoralien>"] = ObjectID::TYPE_PROCTORALIEN;
  goMapping["<minigunalien>"] = ObjectID::TYPE_MINIGUNALIEN;
  goMapping["<predatoralien>"] = ObjectID::TYPE_PREDATORALIEN;
  goMapping["<car>"] = ObjectID::TYPE_CAR;
  goMapping["<civilian>"] = ObjectID::TYPE_CIVILIAN;
  goMapping["<fighter>"] = ObjectID::TYPE_FIGHTER;
  goMapping["<tank>"] = ObjectID::TYPE_TANK;
  goMapping["<playercar>"] = ObjectID::TYPE_PLAYERCAR;
  goMapping["<lightfence>" ] = ObjectID::TYPE_LIGHTFENCE;
  goMapping["<mines>"] = ObjectID::TYPE_MINE;
  goMapping["<youko>"] = ObjectID::TYPE_YOUKO;
  goMapping["<simon>"] = ObjectID::TYPE_SIMON;
  goMapping["<eric>"] = ObjectID::TYPE_ERIC;
  goMapping["<mediumwormalien>"] = ObjectID::TYPE_MEDIUMWORMALIEN;
  goMapping["<flameralien>"] = ObjectID::TYPE_FLAMERALIEN;
  goMapping["<decorative>"] = ObjectID::TYPE_DECORATIVEOBJECT;
  goMapping["<barrel>"] = ObjectID::TYPE_BARREL;
  goMapping["<lightballalien>"] = ObjectID::TYPE_LIGHTBALLALIEN;
  goMapping["<stargate>"] = ObjectID::TYPE_STARGATE;
  goMapping["<machinegun>"] = ObjectID::TYPE_MACHINEGUN;
  goMapping["<guardiantank>"] = ObjectID::TYPE_GUARDIANTANK;
  goMapping["<badassalien>"] = ObjectID::TYPE_BADASSALIEN;
  goMapping["<hiddenwormalien>"] = ObjectID::TYPE_HIDDENWORMALIEN;
  goMapping["<slimealien>"] = ObjectID::TYPE_SLIMEALIEN;
  goMapping["<ultimate_badassalien>"] = ObjectID::TYPE_ULTIMATEBADASSALIEN;
  std::string tmp;
  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    
    rIn >> tmp;
    if ( tmp == "</gameobjects>" ) {
      break;
      
    } else if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    } else {
      std::map< std::string, ObjectID::Type >::iterator it =
          goMapping.find( tmp );
      if ( it == goMapping.end() ) {
        // Unknown gameobject.
        alert("Uknown gameobject", tmp.c_str(), 0, "ok",0, 0,0);
        return -1;
        
      } else {
        // Create the ending tag
        std::string::iterator it2 = tmp.begin() + 1;
        tmp.insert( it2, '/' );
        int ret = readGameObject( it->second, rIn, tmp );
        if ( ret != 0 ) {
          alert("Failed to read GameObject configuration",0,0, "ok",0, 0,0);
          return -1;
        }
      }
    }
  }
  LOG_MESSAGE("            done" );
  return 0;
}



/** Reads the settings for gameobjects
 */
int Settings::readGameObject( ObjectID::Type t, std::istream& rFin,
                              const std::string& rEndTag )
{
  LOG_MESSAGE( "                Reading single object..." );
  Utils::GObjectConfig cfg;
  cfg.objectID = t;
  
  std::string key;
  while ( true ) {
    if ( rFin.eof() == true ) {
      return -1;
    }
    
    rFin >> key;
    if ( key == rEndTag ) {
      gameObjectSettings.push_back( cfg );
      LOG_MESSAGE("                done" );
      return 0;
      
    } else {
      std::string val;
      rFin >> val;
      cfg.properties[ key ] = val;
      
    }
  }
  return -1;
}



/** Initializes the bonusobjects
 */
int Settings::initBonusObjects()
{
  bonusObjectNames.push_back( "RIFLE" );
  bonusObjectNames.push_back( "SHOTGUN" );
  bonusObjectNames.push_back( "BAZOOKA" );
  bonusObjectNames.push_back( "GRENADES" );
  bonusObjectNames.push_back( "CLIP" );
  bonusObjectNames.push_back( "FIRST AID" );
  bonusObjectNames.push_back( "FLAMETHROWER" );
  bonusObjectNames.push_back( "MINIGUN" );
  bonusObjectNames.push_back( "DUAL UZIS" );
}



/** Reads the Weapon Specs
 */
int Settings::readElement_Weapons( std::istream& rIn )
{
  LOG_MESSAGE("            Reading WEAPONS-element....");
  std::string tmp;
  
  // Build the lookup table we could use when converting the weapon-tags
  // from configuration file to integer constants in Weapon - class.
  std::map< std::string, int > weaponMap;
  weaponMap["<rifle>"] = Weapon::W_RIFLE;
  weaponMap["<shotgun>"] = Weapon::W_SHOTGUN;
  weaponMap["<crowbar>"] = Weapon::W_CROWBAR;
  weaponMap["<flamethrower>"] = Weapon::W_FLAMETHROWER;
  weaponMap["<minigun>"] = Weapon::W_MINIGUN;
  weaponMap["<uzis>"] = Weapon::W_UZI;
  weaponMap["<fireball_1>"] = Weapon::W_ALIENFIREBALL;
  weaponMap["<grenade>"] = Weapon::W_GRENADE;
  weaponMap["<tankbullet>"] = Weapon::W_TANKBULLET;
  weaponMap["<sentryfireball>"] = Weapon::W_SENTRYFIREBALL;
  weaponMap["<sniperrifle>"] = Weapon::W_SNIPERRIFLE;
  weaponMap["<rocketlauncher>"] = Weapon::W_ROCKETLAUNCHER;
  
  
  while ( true ) {
    if ( rIn.eof() == true ) {
      alert( "EOF encountered", "WEAPONS-element", 0, "ok",0, 0,0 );
      return -1;
    }
    
    rIn >> tmp;
    if ( tmp == "</weapons>" ) {
      // The end tag
      break;
      
    } else if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    } else {
      // Now we use the lookup table created above to check if we can find
      // the Weapon ID code for the weapon-tag that is placed in 'tmp' -
      // variable.
      std::map< std::string, int >::iterator it = weaponMap.find( tmp );
      if ( it != weaponMap.end() ) {
        // This is familliar tag. No we create the ending tag and load
        // the contents of it.
        std::string::iterator it2 = tmp.begin() + 1;
        tmp.insert( it2, '/' );
        int ret = readWeaponSpecs( it->second, rIn, tmp );
        if ( ret != 0 ) {
          alert( "Failed to read weapon specs", 0,0, "ok",0, 0,0);
          return -1;
        }
      } else {
        alert("Unkown tag in WEAPONS - element", tmp.c_str(), 0, "ok",0, 0,0);
        return -1;
      }
    }
  }
  LOG_MESSAGE("            done" );
  return 0;
}



/** Reads the specs of a weapon
 */
int Settings::readWeaponSpecs( int w, std::istream& rIn,
                               const std::string& rEndTag )
{
  std::string key;
  Utils::WeaponSpecsConfig cfg;
  cfg.weaponTypeID = w;

  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    
    rIn >> key;
    if ( key == rEndTag ) {
      weaponSpecs.push_back( cfg );
      return 0;
      
    } else if ( key == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    } else {
      std::string val;
      rIn >> val;
      cfg.properties[ key ] = val;
      
    }
  }
}

} // end of namespace
